/*
*******************************************************************************************************************
*									Demonstration for Nordic nRF905 module
* Description:  This program demonstrates interface between an Atmel AT89C51RB2 mcu running at 3.3V with
*				nRF905_MOD. The PCB employed was AT89S52-gLCD-STK1. Replace the linear regulator with AMS1113-3.3V to
*				obtain a development platform for 8051 at 3.3V Vdd.
*				Connection with nRF905_MOD completed by strip wire via the extension port. Connection
*				is shown in Hardware Declaration section below.
*				This program simply set the RF module to transmit mode and play a short piece of music after packet
*				got transmitted
*
*				THIS PROJECT REQUIRES ANOTHER PROJECT 'nRF905_Rx.Uv2' (the receive side) TO WORK!!!
*
* File:			..\nRF905_Tx.c
* Software:		Keil C (C51) compiler, Eval version 7.09 under uVision2 IDE v2.40
* Programmer: 	John Leung, TechToys Co. (www.TechToys.com.hk)
* Hardware:		AT89S52-gLCD-STK1 PCB, with Atmel AT89C51RB2, 11.0592 MHz crystal running at 3.3V
* Date:			27 Feb 2007
* Version:		1.0
* Programmer:	John Leung
*
* 												All Rights Reserved
*
*******************************************************************************************************************
*/

#include <REG51RB2.H>
#include "delay.h"

#define ON		0
#define OFF		1
#define uchar	unsigned char
#define uint	unsigned int

//Instruction set for the nRF905 SPI interface; refer to page 16 of nRF905 datasheet
#define WC	0x00	//Write configuration register command
#define RC	0x10	//Read configuration register command
#define WTP	0x20	//Wrte TX Payload command
#define RTP	0x21	//Read TX Payload command
#define WTA	0x22	//Write TX Address command
#define RTA	0x23	//Read TX address command
#define RRP	0x24	//Read RX Payload command


typedef struct _RFconfig
{
	uchar n;
	uchar buf[10];	//RF-CONFIG REGISTER (refer to page 20 of nRF905 datasheet)
}RFconfig;


/*
* Initialization of the Register Contents at byte# 0 - 9 as stated on page 20 of nRF905 datasheet
*/
code RFconfig RxTxconfig =
{
	10,
	0x01,			//byte# 0 : set the channel # to 1
	0x0c,			//byte# 1 : 0bxx[AUTO_RETRAN<5>][RX_RED_PWR<4>][PA_PWR<3:2>][HFREQ_PLL<1>][CH_NO[8]<0>]
					//			0b00001100 => 	AUTO_RETRAN=0; 	No retransmission of data packet
					//							RX_RED_PWR=0; 	Normal operation
					//							PA_PWR=11;		+10dBm output power
					//							HFREQ_PLL=0;	chip operating in 433 MHz band
					//							CH_NO[8]=0;		9th bit of CH_NO is zero
	0x44,			//byte# 2 :	TX_AFW[2:0]=100 => 4 byte TX address field width
 					//			RX_AFW[2:0]=100 => 4 byte RX address field width	
	0x20,			//byte# 3 : RX_payload width of 32 bytes
	0x20,			//byte# 4 : TX_payload width of 32 bytes
	0xcc,			//byte# 5 :	RX_ADDRESS byte 0
	0xcc,			//byte# 6 : RX_ADDRESS byte 1
	0xcc,			//byte# 7 : RX_ADDRESS byte 2
	0xcc,			//byte# 8 : RX_ADDRESS byte 3; thus the RX ADDRESS is configured to 0xcccccccc in this example
					//remarks : The default value is 0xE7E7E7E7
	0x58			//byte# 9 : Set the CRC_MODE, CRC_EN, XOF[2:0], UP_CLK_EN, UP_CLK_FREQ[1:0] bit values
					//			In this configuration, we are doing CRC enable for 8 CRC check bit, 16MHz crystal freq, 
					//			and output clock disable
};

/* declaration of Tx and Rx buffers */
uchar data TxBuf[32];
uchar data RxBuf[32];

/* Hardware declaration */
sbit RF_TX_EN 	= P2^6;
sbit RF_TRX_CE 	= P2^5;
sbit RF_PWR_UP 	= P2^4;
sbit RF_MISO	= P2^3;
sbit RF_MOSI	= P2^2;
sbit RF_SCK		= P2^1;
sbit RF_CSN		= P2^0;

sbit RF_AM		= P0^7;
sbit RF_DR		= P0^6;
sbit RF_CD		= P0^5;

sbit BUZZER		= P1^0;

/* Function prototype */
void 	rfInitIO(void);
void 	uartInit(void);
void 	rfConfig905(void);
void 	rfSetTxMode(void);
void 	rfSetRxMode(void);
void 	rfTxPacket(void);
void 	rfRxPacket(void);
void 	spiWr(uchar);
uchar 	spiRd(void);

void 	ToneGen(unsigned int tone, unsigned int duration);


/*
*******************************************************************************************************************
*											TONE GENERATION ROUTINE
* Function:	This function generates a single tone of a certain frequency 'tone' in Hz, for a given period of time
*			'duration' in msec by using Timer 2 in programmable clock out mode
* Argument:	'tone'		frequency of the tone to generate, in Hz
*			'duration'	duration of tone generate in millisec.
* Return:	none
*
* Note:		*** Please note that there is a limit in the frequency range. We are using a crystal frequency of
*				11.0592MHz. From equal Clock-out Freq= Oscillator Freq / (4*(65536-RCAP2H,RCAP2L)).
*				Thus, max Clock-out Freq = 2.7648MHz; min Clock-out Freq = 42 Hz.
*				Human audible range is around 20Hz - 20kHz, and the frequency response of piezo sounder has 
*				a limited range, too. We just cannot make infinite pitch from our hardware! ***
*******************************************************************************************************************
*/
void ToneGen(unsigned int tone, unsigned int duration)
{
	unsigned long tmp;
	if(tone!=0)
	{
	T2CON = 0x00;			// reset T2CON, the Timer 2 control register. Important for stopping Timer 2 by TR2=0
 
	T2MOD = T2MOD&0xFE;
	T2MOD = T2MOD|0x02;		// sets T2OE=1; DCEN=0; for a clock out mode, no interrupt generated in this mode

	tmp = (11059200UL>>2)/tone;
	tmp = 65536UL - tmp;	
	RCAP2H = (unsigned char) (tmp>>8);
	RCAP2L = (unsigned char) (tmp&0x000000FF);
	TH2 = RCAP2H;
	TL2 = RCAP2L;

	TR2 = 1;				//Start Timer 2 for clock out
	DelayMs(duration);		//software timer here
	TR2 = 0;
	}
	BUZZER = OFF;			//make sure BUZZER is off on exit
}

/*
*********************************************************************************************************
*                         		INITIALIZATION ROUTINE FOR IO PINS (RF)
*
* Description : IO direction setup
*				
* Arguments   : none
*			
* Returns     : none
*********************************************************************************************************
*/
void rfInitIO(void)
{
	RF_CSN = 1;			//chip de-select to start
	RF_SCK = 0;
	RF_DR  = 1;			//set for input mode
	RF_AM  = 1;			//set AM for input mode
	RF_PWR_UP = 1;		//nRF905 module power on
	RF_TRX_CE = 0;		//set nRF905 in standby mode
	RF_TX_EN = 0;		//set radio in receive mode
}

/*
*********************************************************************************************************
*  This is required by initialization routine for the nRF905 module
*********************************************************************************************************
*/
void rfConfig905(void)
{
	uchar i;
	RF_CSN = 0;
	spiWr(WC);
	for(i=0;i<RxTxconfig.n;i++)
	{
		spiWr(RxTxconfig.buf[i]);
	}
	RF_CSN = 1;
}

/*
*********************************************************************************************************
* Configure the module to Tx mode
*********************************************************************************************************
*/
void rfSetTxMode(void)
{
	RF_TX_EN=1;
	RF_TRX_CE=0;
	DelayMs(1);
}


/*
*********************************************************************************************************
* Configure the module to Rx mode
*********************************************************************************************************
*/
void rfSetRxMode(void)
{
	RF_TX_EN=0;
	RF_TRX_CE=1;
	DelayMs(1);
}

/*
*********************************************************************************************************
* Transmit 32 bytes here
*********************************************************************************************************
*/
void rfTxPacket(void)
{
	uchar i;

	RF_CSN=0;
	spiWr(WTP);		//write payload command
	for(i=0;i<32;i++)
	{
		spiWr(TxBuf[i]);
	}
	RF_CSN=1;
	DelayMs(1); 	//short delay
	RF_CSN=0;
	spiWr(WTA);		//write address command
	for(i=0;i<4;i++)
	{
		spiWr(RxTxconfig.buf[i+5]);
	}
	RF_CSN=1;
	RF_TRX_CE=1;	//set TRX_CE high, start Tx data
	DelayUs(20);
	RF_TRX_CE=0;
}

/*
*********************************************************************************************************
* Receive 32 bytes here
*********************************************************************************************************
*/
void rfRxPacket(void)
{
	uchar i;
	
	RF_TRX_CE=0;
	RF_CSN=0;
	spiWr(RRP);
	for(i=0;i<32;i++)
	{
		RxBuf[i]=spiRd();
	}
	RF_CSN=1;
	while(RF_DR||RF_AM);
	RF_TRX_CE=1;
}


/*
*********************************************************************************************************
* Low level SPI WRITE FUNCTION
*********************************************************************************************************
*/
void spiWr(uchar dat)
{
		/* software SPI, send MSB first */
	static uchar i,c;

	c = dat;
	for(i=0;i<8;i++)
	{
	if((c&0x80)==0x80)
		RF_MOSI = 1;
	else
		RF_MOSI = 0;

	RF_SCK = 1;
	c=c<<1;
	RF_SCK = 0;
	}
}
/*
*********************************************************************************************************
* Low level SPI READ FUNCTION
*********************************************************************************************************
*/
uchar spiRd(void)
{
	/* software SPI read, MSB read first */
	static uchar i, dat;

	for(i=0;i<8;i++)
	{
		dat = dat<<1;
		RF_SCK = 1;
		if(RF_MISO)
			dat = dat+1;
		RF_SCK = 0;
	}
	
	return dat;
}


/*
*********************************************************************************************************
*													MAIN
*********************************************************************************************************
*/

void main(void)
{
	unsigned char i;

	rfInitIO();
	rfConfig905();
	rfSetTxMode();

	for(i=0;i<32;i++)				//load data in TxBuf[]
		TxBuf[i] = '0'+i;			//transmit ASCII code from '0'

	while(1)
	{
		rfTxPacket();
		ToneGen(262, 500);	//Do
		ToneGen(294, 500);	//Re
		ToneGen(330, 500);	//Mi
		ToneGen(294, 500);	//Re
		ToneGen(330, 500);	//Mi
		ToneGen(440, 500);	//La
		ToneGen(494, 500);  //Si
	}
}

