/*
*******************************************************************************************************************
*									Demonstration for Nordic nRF905 module
* Description:  This program demonstrates interface between an Atmel AT89C51RB2 mcu running at 3.3V with
*				nRF905_MOD. The PCB employed was AT89S52-gLCD-STK1. Replace the linear regulator with AMS1113-3.3V to
*				obtain a development platform for 8051 at 3.3V Vdd.
*				Connection with nRF905_MOD completed by strip wire via the extension port. Connection
*				is shown in Hardware Declaration section below.
*				This program simply set the RF module to receive mode. Connect a PC COM PORT to J2 of the development
*				board, launch Docklight at 9600bps to monitor the messages upon successful receive of data from nRF905
*				The transmit side can be done by the same development board with Tx mode configured.
* File:			..\nRF905demo\main_Rx.c
* Software:		Keil C (C51) compiler, Eval version 7.09 under uVision2 IDE v2.40
* Programmer: 	John Leung, TechToys Co. (www.TechToys.com.hk)
* Hardware:		AT89S52-gLCD-STK1 PCB, with Atmel AT89C51RB2, 11.0592 MHz crystal running at 3.3V
* Date:			27 Feb 2007
* Version:		1.0
* Programmer:	John Leung
*
* 												All Rights Reserved
*
*******************************************************************************************************************
*/

#include <REG51RB2.H>
#include <intrins.h>
#include "delay.h"
#include <stdio.h>

#define uchar	unsigned char
#define uint	unsigned int

//Instruction set for the nRF905 SPI interface; refer to page 16 of nRF905 datasheet
#define WC	0x00	//Write configuration register command
#define RC	0x10	//Read configuration register command
#define WTP	0x20	//Wrte TX Payload command
#define RTP	0x21	//Read TX Payload command
#define WTA	0x22	//Write TX Address command
#define RTA	0x23	//Read TX address command
#define RRP	0x24	//Read RX Payload command


typedef struct _RFconfig
{
	uchar n;
	uchar buf[10];	//RF-CONFIG REGISTER (refer to page 20 of nRF905 datasheet)
}RFconfig;

/*
* Initialization of the Register Contents at byte# 0 - 9 as stated on page 20 of nRF905 datasheet
*/
code RFconfig RxTxconfig =
{
	10,
	0x01,			//byte# 0 : set the channel # to 1
	0x0c,			//byte# 1 : 0bxx[AUTO_RETRAN<5>][RX_RED_PWR<4>][PA_PWR<3:2>][HFREQ_PLL<1>][CH_NO[8]<0>]
					//			0b00001100 => 	AUTO_RETRAN=0; 	No retransmission of data packet
					//							RX_RED_PWR=0; 	Normal operation
					//							PA_PWR=11;		+10dBm output power
					//							HFREQ_PLL=0;	chip operating in 433 MHz band
					//							CH_NO[8]=0;		9th bit of CH_NO is zero
	0x44,			//byte# 2 :	TX_AFW[2:0]=100 => 4 byte TX address field width
 					//			RX_AFW[2:0]=100 => 4 byte RX address field width	
	0x20,			//byte# 3 : RX_payload width of 32 bytes
	0x20,			//byte# 4 : TX_payload width of 32 bytes
	0xcc,			//byte# 5 :	RX_ADDRESS byte 0
	0xcc,			//byte# 6 : RX_ADDRESS byte 1
	0xcc,			//byte# 7 : RX_ADDRESS byte 2
	0xcc,			//byte# 8 : RX_ADDRESS byte 3; thus the RX ADDRESS is configured to 0xcccccccc in this example
					//remarks : The default value is 0xE7E7E7E7
	0x58			//byte# 9 : Set the CRC_MODE, CRC_EN, XOF[2:0], UP_CLK_EN, UP_CLK_FREQ[1:0] bit values
					//			In this configuration, we are doing CRC enable for 8 CRC check bit, 16MHz crystal freq, 
					//			and output clock disable
};

/* declaration of Tx and Rx buffers */
uchar data TxBuf[32];
uchar data RxBuf[32];

/* Hardware declaration */
sbit RF_TX_EN 	= P2^6;
sbit RF_TRX_CE 	= P2^5;
sbit RF_PWR_UP 	= P2^4;
sbit RF_MISO	= P2^3;
sbit RF_MOSI	= P2^2;
sbit RF_SCK		= P2^1;
sbit RF_CSN		= P2^0;

sbit RF_AM		= P0^7;
sbit RF_DR		= P0^6;
sbit RF_CD		= P0^5;

/* Function prototype */
void 	rfInitIO(void);
void 	uartInit(void);
void 	rfConfig905(void);
void 	rfSetTxMode(void);
void 	rfSetRxMode(void);
void 	TxPacket(void);
void 	RxPacket(void);
void 	spiWr(uchar);
uchar 	spiRd(void);

/*
*********************************************************************************************************
*                         		INITIALIZATION ROUTINE FOR IO PINS (RF)
*
* Description : IO direction setup
*				
* Arguments   : none
*			
* Returns     : none
*********************************************************************************************************
*/
void rfInitIO(void)
{
	RF_CSN = 1;			//chip de-select to start
	RF_SCK = 0;
	RF_DR  = 1;			//set for input mode
	RF_AM  = 1;			//set AM for input mode
	RF_PWR_UP = 1;		//nRF905 module power on
	RF_TRX_CE = 0;		//set nRF905 in standby mode
	RF_TX_EN = 0;		//set radio in receive mode
}

/*
*********************************************************************************************************
*  This is required by initialization routine for the nRF905 module
*********************************************************************************************************
*/
void 	rfConfig905(void)
{
	uchar i;
	RF_CSN = 0;
	spiWr(WC);
	for(i=0;i<RxTxconfig.n;i++)
	{
		spiWr(RxTxconfig.buf[i]);
	}
	RF_CSN = 1;
}

/*
*********************************************************************************************************
*                         					UART INITIALIZATION
*
* Description : This function initialize the UART port of 89S52. Timer 1 is used as the baud rate
*				generator with an external crystal frequency of 11.0592MHz onboard.
*				Baud rate obtained is 9600bps
* Arguments   : none
*			
* Returns     : none
*********************************************************************************************************
*/
void uartInit(void)
{
	SCON  = 0x52;    	//UART in 8 bit mode
 	TMOD  = TMOD|0x20;  //Timer 1 configured for 8-bit auto-reload timer mode, Timer 0 unchanged
 	TH1   = 0xfd;    	//TH1 value for a baud rate 9600bps
	TR1   = 1;			//Timer 1 is turned ON
}

/*
*********************************************************************************************************
* Configure the module to Tx mode
*********************************************************************************************************
*/
void rfSetTxMode(void)
{
	RF_TX_EN=1;
	RF_TRX_CE=0;
	DelayMs(1);
}


/*
*********************************************************************************************************
* Configure the module to Rx mode
*********************************************************************************************************
*/
void rfSetRxMode(void)
{
	RF_TX_EN=0;
	RF_TRX_CE=1;
	DelayMs(1);
}

/*
*********************************************************************************************************
* Transmit 32 bytes here
*********************************************************************************************************
*/
void TxPacket(void)
{
	uchar i;

	RF_CSN=0;
	spiWr(WTP);		//write payload command
	for(i=0;i<32;i++)
	{
		spiWr(TxBuf[i]);
	}
	RF_CSN=1;
	DelayMs(1); 	//short delay
	RF_CSN=0;
	spiWr(WTA);		//write address command
	for(i=0;i<4;i++)
	{
		spiWr(RxTxconfig.buf[i+5]);
	}
	RF_CSN=1;
	RF_TRX_CE=1;	//set TRX_CE high, start Tx data
	DelayUs(20);
	RF_TRX_CE=0;
}

/*
*********************************************************************************************************
* Receive 32 bytes here
*********************************************************************************************************
*/
void RxPacket(void)
{
	uchar i;
	
	RF_TRX_CE=0;
	RF_CSN=0;
	spiWr(RRP);
	for(i=0;i<32;i++)
	{
		RxBuf[i]=spiRd();
	}
	RF_CSN=1;
	while(RF_DR||RF_AM);
	RF_TRX_CE=1;
}


/*
*********************************************************************************************************
* Low level SPI WRITE FUNCTION
*********************************************************************************************************
*/
void spiWr(uchar dat)
{
		/* software SPI, send MSB first */
	static uchar i,c;

	c = dat;
	for(i=0;i<8;i++)
	{
	if((c&0x80)==0x80)
		RF_MOSI = 1;
	else
		RF_MOSI = 0;

	RF_SCK = 1;
	c=c<<1;
	RF_SCK = 0;
	}
}
/*
*********************************************************************************************************
* Low level SPI READ FUNCTION
*********************************************************************************************************
*/
uchar spiRd(void)
{
	/* software SPI read, MSB read first */
	static uchar i, dat;

	for(i=0;i<8;i++)
	{
		dat = dat<<1;
		RF_SCK = 1;
		if(RF_MISO)
			dat = dat+1;
		RF_SCK = 0;
	}
	
	return dat;
}


/*
*********************************************************************************************************
*													MAIN
*********************************************************************************************************
*/

void main(void)
{
	unsigned char i;

	rfInitIO();
	rfConfig905();
	uartInit();					//achieve a baud rate of 9600 bps
	printf("Program Started");	//connect the COM PORT to J2(DB9F) of AT89S52-gLCD-STK1 for monitoring
	rfSetRxMode();

	while(1)
	{
		//This program takes care of the Rx side, printing the whole 32 bytes in Rx PayLoad
		if(RF_DR==1) {
			RxPacket();
			printf("received packet in main()\n");
			for(i=0;i<32;i++)
			printf("Character: %bu \n", RxBuf[i]);
		}
	}
}

